﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Data;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Windows.Forms;
using MFLib;

namespace findImage
{
    // Susieのカタログを扱うクラス
    // 最初にコンストラクタでファイル名をセットした後は、
    // NextThumbの戻り値がfalseになるまで、
    // NextThumbを繰り返して好きな数だけサムネイルを取得する。
    //
    // たとえば、こんな感じに使う
    // SusieCat s = new SusieCat( "hoge.sue" );
    // while( s.NextThumb() ) {
    //      Bitmap bmp = s.GetBitmap();
    // }
    //
    public class SusieCat : IDisposable
    {
        // 内部動作用
        bool isSusieCatalog = false;
        FileStream rfs = null;
        BinaryReader br = null;
        public string catalogFname { get; private set; }

        // サムネイル関連
        public long length { get; private set; }
        int iconType;
        public int bitDepth { get; private set; }
        public int width { get; private set; }
        public int height { get; private set; }
        public UInt32 time { get; private set; }
        int fnameLength;
        public string fname { get; private set; }
        public byte[] image;    // 画像データはBGRの点順次

        // コンストラクタ
        // ヘッダの解析までやってしまう
        public SusieCat(string fname)
        {
            this.catalogFname = fname;
            try
            {
                rfs = new FileStream(fname, FileMode.Open, FileAccess.Read, FileShare.Read);
                br = new BinaryReader(rfs);

                // ヘッダの確認
                if (SuecatHeader(br)) isSusieCatalog = true;
            }
            catch (Exception) { }
        }

        // 後始末
        public void Dispose()
        {
            if (br != null) br.Close();
            if (rfs != null) rfs.Close();
        }

        // susieカタログヘッダがあるときtrueを返す
        private bool SuecatHeader(BinaryReader br)
        {
            // ヘッダ
            if (new string(br.ReadChars(6)) != "SUECAT") return false;

            // ヘッダの終わりを探す
            while (br.ReadByte() != 0xF2) ;
            if (br.ReadByte() != 0xF7) return false;
            if (br.ReadByte() != 0x00) return false;
            return true;
        }

        public long CurrentPosition()
        {
            return rfs.Position;
        }

        // サムネイルのヘッダ部分の本当に頭の部分だけを１つ読み込む
        // もうサムネイルがないときはfalseを返す
        // 高速化のためには
        //      ReadHeader()
        //      ReadBody()
        //      SkipBody()
        //          この３つを正しく組み合わせて使う
        // 普通の用途には
        //      ReadNextThumb()
        //          これを使うのが安全
        long startPos;
        public bool ReadHeader()
        {
            if (!isSusieCatalog) return false;
            image = null;
            try
            {
                // 未対応サムネイルをスキップするための
                // gotoの代わりの嘘ループ
                while (true)
                {
                    // (ほぼ)終わりまで読んだ
                    if (rfs.Position > rfs.Length - 10)
                    {
                        isSusieCatalog = false;
                        return false;
                    }

                    // 個々のサムネイルのヘッダ
                    startPos = br.BaseStream.Position;
                    length = br.ReadInt32();
                    iconType = br.ReadByte();
                    if (iconType != 1)
                    {
                        SkipBody();
                        continue;
                    }

                    bitDepth = br.ReadByte();
                    if (bitDepth != 16 && bitDepth != 24)
                    {
                        SkipBody();
                        continue;
                    }

                    width = br.ReadInt16();
                    height = br.ReadInt16();
                    break;
                } // while(true)
            }
            catch (Exception)
            {
                isSusieCatalog = false;
                return false;
            } // catch
            return true;
        }

        // ReadHeaderの続きを読む
        public void ReadBody()
        {
            time = br.ReadUInt32();
            fnameLength = br.ReadByte();
            byte[] fnameByte = new byte[fnameLength];
            fnameByte = br.ReadBytes(fnameLength);
            fname = Encoding.GetEncoding("SHIFT_JIS").GetString(fnameByte);

            image = new byte[3 * height * width];
            switch (bitDepth)
            {
                case 24:
                    image = br.ReadBytes(width * height * 3);   // B,G,Rの点順次。画面上から下
                    break;
                case 16:
                    int pos = 0;
                    for (int y = 0; y < height; y++)
                    {
                        for (int x = 0; x < width; x++)
                        {
                            int a = br.ReadByte();
                            int b = br.ReadByte();

                            image[pos++] = (byte)((a >> 3) << 3);
                            image[pos++] = (byte)((((a & 7) << 3) + (b >> 5)) << 2);
                            image[pos++] = (byte)((b & 31) << 3);
                        }
                    } // for
                    break;
            } // switch
        }

        // ReadHerderの続きをスキップする
        public void SkipBody()
        {
            br.BaseStream.Seek(startPos + length, SeekOrigin.Begin);
        }
        
        // 次のサムネイルを読む
        // もう無いときはfalseを返す
        public bool NextThumb()
        {
            if (ReadHeader())
            {
                ReadBody();
                return true;
            }
            else
            {
                return false;
            }
        }

        // このカタログファイルは、アーカイブなのか、ディレクトリなのか
        public bool isArchives()
        {
            return isArchives(catalogFname);
        }
        static bool isArchives(string fname)
        {
            return !(Path.GetFileName(fname).ToUpper() == "_THUMBNL.SUE");
        }

        // bitmapを取得する。
        // 高速化するには、LockBitsを使うらしい。
        // しかし、bitmapの取得回数は多くないだろうから、このままでいいや
        public Bitmap getBitmap()
        {
            Bitmap bitmap = new Bitmap(width, height);
            ImageProcess.resizeImage(image, width, height, width * 3, bitmap);
            ImageProcess.flipud(bitmap);
            return bitmap;
        }

        // サムネイル画像のフルパス名を得る
        // サムネイルファイルが_THUMBNL.SUEの時は、正しいフルパス名が得られるが
        // それ以外の場合は、何らかのアーカイブ内のファイル、ということになる。
        // 開けるかどうかは別にして、アーカイブ名のディレクトリがある、と仮定する。
        public static string GetThumbFullPathName(string sueFname, string thumbFname)
        {
            string fullPathName;
            if (isArchives(sueFname))
            {
                // アーカイブ(zip, lzh, psd, etc)のサムネイルファイルの場合
                fullPathName = Path.GetDirectoryName(sueFname) + "\\" + Path.GetFileNameWithoutExtension(sueFname) + "\\" + thumbFname;
            }
            else
            {
                // ディレクトリのサムネイルファイルの場合
                fullPathName = Path.GetDirectoryName(sueFname) + "\\" + thumbFname;
            }
            return fullPathName;
        }

    }

}
