﻿using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Windows.Forms;
using MFLib;

namespace findImage
{

    // 検索を管理・実行するクラス
    public class Search
    {
        MainWindow mainWindow;
        TabControl tabControl;
        Queue<string> queFolder;    // サブフォルダ探索用(再帰ではresumeを実現できないためFIFOを使う)
        public bool canClear { private set; get; }

        // サイズを高速で照合するためのクラス
        // サイズは全てのサムネイルと照合するため
        // それ専用のデータを作って高速化を図る
        public class SizeData
        {
            public int width;
            public int height;
            public MyTabPage tabPage;
            public int n;
            public SizeData next;
        }
        SizeData sizeList;

        private Search()
        {
            queFolder = new Queue<string>();
            canClear = false;
        }

        static Search instance = null;
        public static Search GetInstance()
        {
            if (instance == null)
            {
                instance = new Search();
            }
            return instance;
        }

        
        // 検索開始ディレクトリも指定
        public void Init(MainWindow mainWindow, TabControl tabControl, string searchFolder)
        {
            queFolder.Clear();
            queFolder.Enqueue(searchFolder);

            Init(mainWindow, tabControl);
        }

        // 前回の続き
        public void Init(MainWindow mainWindow, TabControl tabControl)
        {
            this.mainWindow = mainWindow;
            this.tabControl = tabControl;

            // サイズの高速照合用にSizeListの単方向リストを作成する。
            sizeList = null;
            for (int idx = tabControl.Controls.Count - 2; idx >= 0; idx--)
            {
                MyTabPage tp = (MyTabPage)tabControl.Controls[idx];
                SearchData sd = tp.searchData;
                if (!sd.available) continue;    // 中には、検索画像を設定していないタブだってある
                for (int i = 0; i < sd.THUMBNAIL_SIZE.Length; i++)
                {
                    SizeData p = new SizeData();
                    p.width = sd.width[i];
                    p.height = sd.height[i];
                    p.tabPage = tp;
                    p.n = i;
                    p.next = sizeList;
                    sizeList = p;
                }
            }
        }

        // 検索結果をクリア
        public void Clear()
        {
            if (!canClear) return;
            for (int i = 0; i < tabControl.Controls.Count - 1; i++)
            {
                MyTabPage tabPage = (MyTabPage)tabControl.Controls[i];
                tabPage.Clear();
            }
            canClear = false;
        }


        // ■■検索スレッド■■■■■■■■■■■■■■■■■■■■
        // ある画像との相関を求め、順位を更新する
        BackgroundWorker bgWorker;
        DoWorkEventArgs bgEvent;

        public void Run(object sender, DoWorkEventArgs e)
        {
            canClear = true;
            bgWorker = sender as BackgroundWorker;
            bgEvent = e;

            SearchFolder();
        }

        // 指定ディレクトリ以下の全ての.sueファイルを検索
        private void SearchFolder()
        {
            while (queFolder.Count > 0)
            {
                if (isCancel()) return;

                string folder = queFolder.Peek();
                InvokeSetSearchingDir(folder);

                // .sueファイルを探して処理
                // いつでも中断可能
                try
                {
                    string[] files = Directory.GetFiles(folder);
                    foreach (string fname in files)
                    {
                        if (isCancel()) return;
                        if (Path.GetExtension(fname).ToLower() == ".sue")
                        {
                            // 拡張子がsueのものについて実行
                            if (CheckCatalog(fname)) return;
                        }
                    }
                }
                catch (Exception) { }

                // サブディレクトリを検索
                // FIFOがこわれるので、この間は中断禁止
                // 双方向リストで実装して、元に戻すようにすればいいんだけど、
                // そこまですることはないでしょう
                try
                {
                    string[] dirs = Directory.GetDirectories(folder);
                    foreach (string dname in dirs)
                    {
                        queFolder.Enqueue(dname);
                    }
                }
                catch (Exception) { }
                // このディレクトリに関する処理は終わった
                queFolder.Dequeue();
            }

            // おわった
            bgWorker.ReportProgress(100);
        }

        // susieカタログファイルを処理
        // 戻り値
        //  true    GUIから中断を要請された
        //  false   このファイルの処理を終了した
        private bool CheckCatalog(string fname)
        {
            var fileInfo = new FileInfo(fname);
            long fileLength = fileInfo.Length;

            //Console.WriteLine("check = " + fname);
            using (SusieCat susieCat = new SusieCat(fname))
            {
                // カタログ内の(対応している)全サムネイルについて
                while (susieCat.ReadHeader())
                {
                    if (isCancel()) return true;

                    int percentage = (int)(susieCat.CurrentPosition() * 100 / fileLength);
                    bgWorker.ReportProgress(percentage);

                    int width = susieCat.width;
                    int height = susieCat.height;
                    bool imageLoaded = false;
                    for (SizeData p = sizeList; p != null; p = p.next)
                    {
                        if (p.width != width || p.height != height) continue;

                        // サイズが等しい時のみ、画像の相関をテストする
                        if (isCancel()) return true;
                        if (!imageLoaded)
                        {
                            imageLoaded=true;
                            susieCat.ReadBody();
                        }
                        double corr = CalcCorrMAE(susieCat.image, p.tabPage.searchData.data2D[p.n], width, height);
                        p.tabPage.CheckImage(corr, susieCat, fname);
                    } // for
                    if (!imageLoaded) susieCat.SkipBody();
                } // while
            } // using
            return false;
        }

        // 相関を求める。差の絶対値の平均。CGならこれで良い。写真だとダメだけど
        private double CalcCorrMAE(byte[] img1, byte[] img2, int width, int height)
        {
            double corr;
            unsafe
            {
                fixed (byte* img1address = img1)
                {
                    fixed (byte* img2address = img2)
                    {
                        int sum = 0;
                        int pixNum = width * height * 3;
                        byte* p1 = img1address;
                        byte* p2 = img2address;
                        for (int i = 0; i < pixNum; i++)
                        {
                            int i1 = *p1++;
                            int i2 = *p2++;

                            if (i1 > i2)
                            {
                                sum += i1 - i2;
                            }
                            else
                            {
                                sum += i2 - i1;
                            }
                        }

                        corr = 1 - (double)(sum) / (255.0 * pixNum);

                    } // fixed p2
                } // fixed p1
            } // unsafe
            return corr;
        }

        // 相関を表示用に変換する
        public static string corr2string(double corr)
        {
            double c2 = Math.Ceiling(corr * 100)/100;

            return Math.Ceiling(c2 * c2 * c2 * 100).ToString() + "%";
        }

        // GUIからキャンセルがあったかどうか
        // 他の変数を参照するようにすれば、
        // キャンセル以外のさまざまなインタラプトに対応できる
        private bool isCancel()
        {
            if (bgWorker.CancellationPending)
            {
                bgEvent.Cancel = true;
                return true;
            }
            return false;
        }

        // ■■GUIスレッドに何かさせる■■■■■■■■■■■■■■■■■■■■
        private void InvokeSetSearchingDir(string str)
        {
            mainWindow.Invoke((MethodInvoker)delegate()
            {
                mainWindow.SetSearchingDir(str);
            });
        }
    }



}
