﻿using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Windows.Forms;
using Microsoft.Win32;
using MFLib;


namespace findImage
{
    [System.ComponentModel.DesignerCategory("")]
    public class MyTabPage : MFTabPage
    {
        MainWindow mainWindow;

        // GUIパーツ
        static int GRID_THUMB = 0;
        static int GRID_CORR = 1;
        static int GRID_PATH = 2;
        MyPictureBox pictureBox = new MyPictureBox();       // 検索する画像を表示する枠
        MyButton buttonPasteImage = new MyButton();         // 「クリップボードから貼り付け」ボタン
        MyLabel labelImageToSearch = new MyLabel();         // 「検索画像」ラベル
        MyTextBox textBoxImageToSearch = new MyTextBox();   // 検索画像のパス
        MyLabel labelSelectedImage = new MyLabel();         // 「選択画像」ラベル
        MyTextBox textBoxSelectedImage = new MyTextBox();   // 選択画像のパス
        MyButton buttonRefImageToSearch = new MyButton();   // 検索画像の「参照…」ボタン
        MyButton buttonOpenSelectedImage = new MyButton();  // 選択画像を「開く」ボタン
        MyButton buttonCloseSearch = new MyButton();        // このパネルを閉じる
        MyDataGridView gridView = new MyDataGridView();     // 候補画像表示枠
        
        // 検索用データ
        public SearchData searchData;       // この画像を探す
        double minCorr;                     // 現時点での最小の相関
        LinkedList<CorrData> corrList;      // 候補リスト
        class CorrData
        {
            public double corr;
            public string fname;
            public string catalogFname;
            public string thumbFname;
            public bool isArchived;
        }

        public MyTabPage(MainWindow mainWindow)
            : base()
        {
            Text = String.Format("検索画像");
            this.mainWindow = mainWindow;
            searchData = new SearchData(mainWindow, this);
            corrList = new LinkedList<CorrData>();
            Clear();

            // 初期状態ではサイズが小さすぎて自動レイアウトがうまく働かないため
            Size = new Size(1000, 1000);

            #region GUIパーツの配置

            // pictureBox
            pictureBox.tabPage = this;
            pictureBox.BorderStyle = BorderStyle.FixedSingle;
            pictureBox.BackColor = Color.Ivory;
            pictureBox.SizeMode = PictureBoxSizeMode.CenterImage;
            //pictureBox.Image = Misc.getBitmapFromResource("findImage.image.drop.bmp");
            pictureBox.TabStop = false;
            pictureBox.mf.SetLocation(this, HorizontalLocation.LeftIn, VerticalLocation.TopIn);
            pictureBox.Size = new Size(98, 98);
            pictureBox.AllowDrop = true;
            pictureBox.DragEnter += new DragEventHandler(pictureBox_DragEnter);
            pictureBox.DragDrop += new DragEventHandler(pictureBox_DragDrop);
            Controls.Add(pictureBox);

            // buttonPasteImage
            buttonPasteImage.tabPage = this;
            buttonPasteImage.Text = "クリップボード内の画像を貼り付け";
            buttonPasteImage.Size = new Size(178, 23);
            buttonPasteImage.mf.SetLocation(pictureBox, HorizontalLocation.RightOf, VerticalLocation.TopSame);
            buttonPasteImage.Click += new EventHandler(buttonPasteImage_Click);
            Controls.Add(buttonPasteImage);

            // labelImagetoSearch
            labelImageToSearch.Text = "検索画像/";
            labelImageToSearch.Size = new Size(53, 12);
            labelImageToSearch.mf.SetLocation(buttonPasteImage, HorizontalLocation.LeftSame, VerticalLocation.BottomOf);
            Controls.Add(labelImageToSearch);

            // buttonRefImageToSearch
            buttonRefImageToSearch.tabPage = this;
            buttonRefImageToSearch.Text = "参照…";
            buttonRefImageToSearch.Size = new Size(75, 23);
            buttonRefImageToSearch.mf.SetLocation(this, HorizontalLocation.RightIn, labelImageToSearch, VerticalLocation.BottomOf);
            buttonRefImageToSearch.Click += new EventHandler(buttonRefImageToSearch_Click);
            Controls.Add(buttonRefImageToSearch);


            // textBoxImageToSearch
            textBoxImageToSearch.tabPage = this;
            textBoxImageToSearch.Text = "検索画像(を含むSusieサムネイル)をここか左の枠にドロップ";
            textBoxImageToSearch.mf.SetLocation(labelImageToSearch, HorizontalLocation.LeftSame, buttonRefImageToSearch, VerticalLocation.CenterOf);
            textBoxImageToSearch.mf.SetWidth(pictureBox, HorizontalLocation.RightOf, buttonRefImageToSearch, HorizontalLocation.LeftOf);
            textBoxImageToSearch.AllowDrop = true;
            textBoxImageToSearch.DragEnter += new DragEventHandler(textBoxImageToSearch_DragEnter);
            textBoxImageToSearch.DragDrop += new DragEventHandler(textBoxImageToSearch_DragDrop);
            textBoxImageToSearch.KeyDown += new KeyEventHandler(textBoxImageToSearch_KeyDown);
            Controls.Add(textBoxImageToSearch);

            // labelSelectedImage
            labelSelectedImage.Text = "選択画像";
            labelSelectedImage.Size = new Size(53, 12);
            labelSelectedImage.mf.SetLocation(textBoxImageToSearch, HorizontalLocation.LeftSame, VerticalLocation.BottomOf);
            Controls.Add(labelSelectedImage);

            // buttonOpenSelectedImage
            buttonOpenSelectedImage.tabPage = this;
            buttonOpenSelectedImage.Text = "開く";
            buttonOpenSelectedImage.Size = new Size(75, 23);
            buttonOpenSelectedImage.mf.SetLocation(this, HorizontalLocation.RightIn, labelSelectedImage, VerticalLocation.BottomOf);
            buttonOpenSelectedImage.Click += new EventHandler(buttonOpenSelectedImage_Click);
            Controls.Add(buttonOpenSelectedImage);

            // textBoxSelectedImage
            textBoxSelectedImage.tabPage = this;
            textBoxSelectedImage.mf.SetLocation(labelSelectedImage, HorizontalLocation.LeftSame, buttonOpenSelectedImage, VerticalLocation.CenterOf);
            textBoxSelectedImage.mf.SetWidth(pictureBox, HorizontalLocation.RightOf, buttonOpenSelectedImage, HorizontalLocation.LeftOf);
            Controls.Add(textBoxSelectedImage);

            // buttonCloseSearch
            buttonCloseSearch.tabPage = this;
            buttonCloseSearch.Text = "x";
            buttonCloseSearch.TextAlign = ContentAlignment.MiddleCenter;
            buttonCloseSearch.Font = new System.Drawing.Font("MS UI Gothic", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((byte)(128)));
            buttonCloseSearch.Size = new Size(20, 18);
            buttonCloseSearch.mf.SetLocation(this, HorizontalLocation.RightIn, VerticalLocation.TopIn);
            buttonCloseSearch.Click += new EventHandler(buttonCloseSearch_Click);
            Controls.Add(buttonCloseSearch);

            // gridView
            //  column
            //      0   画像
            //      1   表示用類似度(%表示)
            //      2   ファイル名
            gridView.tabPage = this;
            Control c = MFExtention.GetAroundControl(pictureBox, textBoxSelectedImage, buttonOpenSelectedImage);
            gridView.mf.SetLocation(c, HorizontalLocation.LeftSame, c, VerticalLocation.BottomOf);
            gridView.mf.SetWidth(c, HorizontalLocation.LeftSame, c, HorizontalLocation.RightSame);
            gridView.mf.SetHeight(c, VerticalLocation.BottomOf, this, VerticalLocation.BottomIn);
            gridView.AllowUserToAddRows = false;
            gridView.AllowUserToResizeColumns = false;
            gridView.AllowUserToResizeRows = false;
            gridView.MultiSelect = false;
            gridView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
            gridView.CellPainting += new DataGridViewCellPaintingEventHandler(gridView_CellPainting);
            gridView.RowHeadersWidth = 40;
            gridView.RowHeadersWidthSizeMode = DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            gridView.RowTemplate.Height = (96 + 1);
            gridView.ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.DisableResizing;
            gridView.ScrollBars = ScrollBars.Vertical;
            gridView.BorderStyle = BorderStyle.FixedSingle; // 1本線
            // 画像列を追加
            {
                DataGridViewImageColumn column = new DataGridViewImageColumn();
                column.ImageLayout = DataGridViewImageCellLayout.Normal;
                gridView.Columns.Add(column);
            }
            // テキスト列を追加
            for (int i = 1; i < 3; i++)
            {
                DataGridViewTextBoxColumn column = new DataGridViewTextBoxColumn();
                gridView.Columns.Add(column);
            }
            gridView.Columns[GRID_THUMB].Name = "サムネイル";
            gridView.Columns[GRID_THUMB].Width = (96 + 1);
            gridView.Columns[GRID_THUMB].SortMode = DataGridViewColumnSortMode.NotSortable;
            gridView.Columns[GRID_CORR].Name = "類似度";
            gridView.Columns[GRID_CORR].Width = 64;
            gridView.Columns[GRID_CORR].SortMode = DataGridViewColumnSortMode.NotSortable;
            gridView.Columns[GRID_PATH].Name = "ファイル名";
            gridView.Columns[GRID_PATH].SortMode = DataGridViewColumnSortMode.NotSortable;
            gridView.Columns[GRID_PATH].DefaultCellStyle.WrapMode = DataGridViewTriState.True;  // 折り返して表示
            gridView.Columns[GRID_PATH].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
            gridView.Click += new EventHandler(gridView_Click);
            gridView.DoubleClick += new EventHandler(gridView_DoubleClick);
            Controls.Add(gridView);
            #endregion

        }

        #region GUIパーツ関連の関数
        // gridViewのクリック
        void gridView_Click(object sender, EventArgs e)
        {
            textBoxSelectedImage.Text = GetCurrentSelectedFname();

        }

        // gridViewのダブルクリック
        void gridView_DoubleClick(object sender, EventArgs e)
        {
            OpenSelectedImage();
        }

        // 開くボタン
        void buttonOpenSelectedImage_Click(object sender, EventArgs e)
        {
            OpenSelectedImage();
        }

        // クリップボードから画像貼り付けボタン
        private void buttonPasteImage_Click(object sender, EventArgs e)
        {
            if (mainWindow.IsBusy()) return;

            IDataObject data = Clipboard.GetDataObject();
            if (data.GetDataPresent(DataFormats.Bitmap))
            {
                Bitmap bmp = (Bitmap)data.GetData(DataFormats.Bitmap);
                ((MyButton)sender).tabPage.SetImage(bmp, null);
            }
        }

        // textBoxImageToSearchでenterキーが押されたときの処理
        private void textBoxImageToSearch_KeyDown(object sender, KeyEventArgs e)
        {
            if (mainWindow.IsBusy()) return;

            if (e.KeyCode == Keys.Enter)
            {
                string[] fnames = new string[1];
                fnames[0]=textBoxImageToSearch.Text;
                EntrySearchFiles(((MyTextBox)sender).tabPage, fnames);
            }
        }

        // クローズボタン
        // 最後の検索画面を閉じると、自動的に新しい検索画面が追加される。
        // その理由:
        // 検索画面を閉じると、tabControl_SelectedIndexChangedが発生する。
        // 最後の検索画面を閉じた時は、「他画像の検索」が自動的に選択され
        // その結果、新しい検索画面が追加される。
        private void buttonCloseSearch_Click(object sender, EventArgs e)
        {
            if (mainWindow.IsBusy()) return;

            mainWindow.DeleteTab(((MyButton)sender).tabPage);
        }

        // 参照ボタン
        private void buttonRefImageToSearch_Click(object sender, EventArgs e)
        {
            if (mainWindow.IsBusy()) return;

            var dlg = new OpenFileDialog();
            dlg.Title = "検索する画像を選択してください";
            dlg.RestoreDirectory = true;
            dlg.Multiselect = true;
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                EntrySearchFiles(((MyButton)sender).tabPage, dlg.FileNames);
            }
        }

        // pictureBoxにドラッグ中
        private void pictureBox_DragEnter(object sender, DragEventArgs e)
        {
            if (mainWindow.IsBusy()) return;
            e.Effect = DragDropEffects.All;
        }

        // pictureBoxにファイルがドロップされた
        private void pictureBox_DragDrop(object sender, DragEventArgs e)
        {
            if (mainWindow.IsBusy()) return;
            DoDropFiles(((MyPictureBox)sender).tabPage, e);
        }

        // textBoxImageToSearchにドラッグ中
        private void textBoxImageToSearch_DragEnter(object sender, DragEventArgs e)
        {
            if (mainWindow.IsBusy()) return;
            e.Effect = DragDropEffects.All;
        }

        // textBoxImageToSearchにファイルがドロップされた
        private void textBoxImageToSearch_DragDrop(object sender, DragEventArgs e)
        {
            if (mainWindow.IsBusy()) return;
            DoDropFiles(((MyTextBox)sender).tabPage, e);
        }

        // ドロップされた検索ファイル群の処理
        private void DoDropFiles(MyTabPage tabPage, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                EntrySearchFiles(tabPage, (string[])e.Data.GetData(DataFormats.FileDrop));
            }
        }

        // 検索する(複数の)画像ファイルを設定する
        // 引数
        //  tabPage     処理対象のtabPage
        //  fnames      検索用に選んだファイル群
        private void EntrySearchFiles(MyTabPage tabPage, string[] fnames)
        {
            foreach (string fname in fnames)
            {
                // susieのカタログを指定されたときは、そこから画像を選ぶ
                if (Path.GetExtension(fname).ToLower() == ".sue")
                {
                    using (var selectForm = new SelectFilesFromSusieCatalog())
                    {
                        selectForm.SetSusieCatalog(fname);
                        selectForm.ShowDialog();

                        Bitmap bmp;
                        string thumb_name;
                        while (selectForm.GetNextImage(out bmp, out thumb_name))
                        {
                            EntryBitmap(tabPage, bmp, SusieCat.GetThumbFullPathName(fname, thumb_name));
                        }
                    } // using
                }
                else
                {
                    try
                    {
                        Bitmap bmp = new Bitmap(fname);
                        EntryBitmap(tabPage, bmp, fname);
                    }
                    catch (Exception)
                    {
                        continue;
                    }
                }
            }
        }

        // bitmapを設定する
        private void EntryBitmap(MyTabPage tabPage, Bitmap bmp, string fname)
        {
            if (!searchData.available)
            {
                // このタブにはまだ検索画像が設定されていない
                tabPage.SetImage(bmp, fname);
            }
            else
            {
                // 新しいタブに画像を設定
                MyTabPage newTab = mainWindow.AddNewTab();
                newTab.SetImage(bmp, fname);
            }
        }

        // ボタンの外見設定
        public void ManageButtonEnable()
        {
            if (mainWindow.IsBusy())
            {
                buttonPasteImage.Enabled = false;
                buttonRefImageToSearch.Enabled = false;
                buttonCloseSearch.Enabled = false;
                pictureBox.AllowDrop = false;
                textBoxImageToSearch.AllowDrop = false;
            }
            else
            {
                buttonPasteImage.Enabled = true;
                buttonRefImageToSearch.Enabled = true;
                buttonCloseSearch.Enabled = true;
                pictureBox.AllowDrop = true;
                textBoxImageToSearch.AllowDrop = true;
            }
        }

        // 行番号の表示。以下を参照
        // http://dobon.net/vb/dotnet/datagridview/drawrownumber.html
        private void gridView_CellPainting(object sender, DataGridViewCellPaintingEventArgs e)
        {
            //列ヘッダーかどうか調べる
            if (e.ColumnIndex < 0 && e.RowIndex >= 0)
            {
                //セルを描画する
                e.Paint(e.ClipBounds, DataGridViewPaintParts.All);

                //行番号を描画する範囲を決定する
                //e.AdvancedBorderStyleやe.CellStyle.Paddingは無視しています
                Rectangle indexRect = e.CellBounds;
                indexRect.Inflate(-2, -2);  // borderの分を引く
                //行番号を描画する
                TextRenderer.DrawText(e.Graphics,
                    (e.RowIndex + 1).ToString(),
                    e.CellStyle.Font,
                    indexRect,
                    e.CellStyle.ForeColor,
                    TextFormatFlags.Right | TextFormatFlags.VerticalCenter);
                //描画が完了したことを知らせる
                e.Handled = true;
            }
        }
        #endregion

        // 現在選択中のセルのファイル名を得る
        string GetCurrentSelectedFname()
        {
            var cell = gridView.CurrentCell;
            if (cell == null) return null;
            return (string)gridView[2, cell.RowIndex].Value;
        }

        // 現在選択中のセルの詳細(CorrData)を得る
        // nullを返す場合がある
        //  要素が増えるときは「先にcorrListに追加してからgridViewに追加」
        //  要素を減らすときは「先にgridViewから削除してから、corrListから削除」の順番だから、
        //  普通は、こういう事態にはならない。
        //  しかし、GUIに指示を出してから実際に反映されるまでの間に
        //  クリックが行われた場合に「セルはあるのに、corrListからは削除されている」
        //  という事態が起こりうる。このとき、nullを返す
        CorrData GetCurrentSelectedCorrData()
        {
            var cell = gridView.CurrentCell;
            if (cell == null) return null;
            int idx = cell.RowIndex;
            var p = corrList.First;
            for (int i = 0; i < idx && p != null; i++)
            {
                p = p.Next;
            }
            return (p != null) ? p.Value : null;
        }

        // 選択中の画像を開く
        // アーカイブ内のファイルも開くために、非常に執念深い作りになっている…
        void OpenSelectedImage()
        {
            string fname = GetCurrentSelectedFname();
            if (fname == null) return;
            
            try
            {
                // 普通のファイルなら、これでおしまい
                if (mainWindow.susiePath != "")
                {
                    System.Diagnostics.Process.Start(mainWindow.susiePath, fname);
                }
                else
                {
                    System.Diagnostics.Process.Start(fname);
                }
            }
            catch (Exception)
            {
                // 一発で開けなかった場合は
                // ファイルが存在しない場合と
                // ファイルにアプリケーションが関連づけされていない場合と、
                // アーカイブ内のファイルを開こうとした場合がある
                CorrData c = GetCurrentSelectedCorrData();
                if (c == null) return;          // GUIのタイミングによっては起こりうる。

                if (!c.isArchived)
                {
                    // この場合はファイルが消されたか対応付けされていない
                    if (!File.Exists(fname))
                    {
                        mainWindow.DrawMessage("\"" + fname + "\"が存在しません。\n");
                    }
                    else if (mainWindow.susiePath != null && !File.Exists(mainWindow.susiePath))
                    {
                        mainWindow.DrawMessage("画像ビューワに指定した\"" + mainWindow.susiePath + "\"が存在しません。\n");
                    }
                    else
                    {
                        mainWindow.DrawMessage("何らかの理由で\"" + fname + "\"を開けません。\n");
                    }
                    return;
                }

                // ここに来たということは、アーカイブ内のファイルを開こうとしている
                mainWindow.DrawMessage("\"" + fname + "\"は、アーカイブ内ファイルです。開けないかもしれません。\n");

                // まず、ファイルに関連付けされているアプリケーションを得る
                string commandPath;
                string commandArg;
                if (mainWindow.susiePath != "")
                {
                    // (susieの)パスが設定されている場合
                    commandPath = mainWindow.susiePath;
                    commandArg = "{0}";
                }
                else
                {
                    // 設定されていない場合は自分で調べる
                    string ext = Path.GetExtension(fname);
                    try
                    {
                        Misc.GetProgramFromExt(ext, out commandPath, out commandArg);
                    }
                    catch (Exception)
                    {
                        mainWindow.DrawMessage("拡張子 " + ext + " に関連付けられているプログラムが解りません。\n");
                        return;
                    }
                }
             
                // 元のアーカイブの候補を探して、順番に開いていく
                string dirName = Path.GetDirectoryName(c.catalogFname);
                string searchPattern = Path.GetFileNameWithoutExtension(c.catalogFname)+".*";
                try
                {
                    string[] files = Directory.GetFiles(dirName, searchPattern);
                    foreach (var archiveCandName in files)
                    {
                        if (Path.GetExtension(archiveCandName).ToLower() == ".sue") continue;
                        string openFname = archiveCandName + "\\" + c.thumbFname;
                        try
                        {
                            System.Diagnostics.Process.Start(commandPath, String.Format(commandArg, openFname));
                        }
                        catch (Exception) { }
                    }
                }
                catch (Exception) { }
                

            } // catch
        }


        // 検索する画像の、検索用画像データ作成および表示用サムネイル作成と表示
        private void SetImage(Bitmap bmp, string fname)
        {
            searchData.SetImage(bmp);
            pictureBox.Image = searchData.thumbnail;
            textBoxImageToSearch.Text = fname;
        }

        // 検索結果の初期化
        public void Clear()
        {
            gridView.Rows.Clear();
            minCorr = 0;
            corrList.Clear();
            Invalidate();
        }

        // ■■検索スレッド■■■■■■■■■■■■■■■■■■■■
        // 画像の追加をチェックする
        public void CheckImage(double corr, SusieCat susieCat, string catalogFname)
        {
            // 既にリストに含まれているか？
            foreach (var c in corrList)
            {
                if (c.thumbFname == susieCat.fname && c.catalogFname == catalogFname) return;
            }

            // 既に最大数に達しているか？
            if (corrList.Count >= mainWindow.maxCandidate)
            {
                if (corr < minCorr)
                {
                    return;
                }
                else
                {
                    // 最後の１つを消す
                    InvokeRemoveRow(gridView, corrList.Count - 1);
                    corrList.RemoveLast();
                }
            }

            // 挿入すべき位置を探す
            int i = 0;
            var p = corrList.First;
            while (i < corrList.Count)
            {
                if (corr >= p.Value.corr) break;
                p = p.Next;
                i++;
            }

            // 挿入
            string fname = SusieCat.GetThumbFullPathName(catalogFname, susieCat.fname);
            InvokeInsertRow(gridView, i, susieCat.getBitmap(), Search.corr2string(corr), fname);
            var corrData = new CorrData();
            corrData.corr = corr;
            corrData.fname = fname;
            corrData.catalogFname = catalogFname;
            corrData.thumbFname = susieCat.fname;
            corrData.isArchived = susieCat.isArchives();

            if (i == 0)
            {
                corrList.AddFirst(corrData);
            }
            else if (i == corrList.Count)
            {
                corrList.AddLast(corrData);
            }
            else
            {
                corrList.AddBefore(p, corrData);
            }

            // 最小相関値を更新
            minCorr = corrList.Last.Value.corr;
        }


        // ■■検索スレッドからGUIスレッドに何かさせる■■■■■■■■■■■■■■■■■■■■
        private void InvokeRemoveRow(DataGridView gridView, int N)
        {
            mainWindow.Invoke((MethodInvoker)delegate()
            {
                gridView.Rows.RemoveAt(N);
            });
        }

        private void InvokeInsertRow(DataGridView gridView, int N, Bitmap bmp, string corr, string fname )
        {
            mainWindow.Invoke((MethodInvoker)delegate()
            {
                gridView.Rows.Insert(N, 1);
                gridView[GRID_THUMB, N].Value = bmp;
                gridView[GRID_CORR, N].Value = corr;
                gridView[GRID_PATH, N].Value = fname;
            });
        }
    } // class

    // ■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
    // そのほかのクラス 
    #region  tabPageを追加しているだけ
    [System.ComponentModel.DesignerCategory("")]
    public class MyLabel : MFLabel
    {
        public MyTabPage tabPage;             // 自分が属しているTabPage
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MyButton : MFButton
    {
        public MyTabPage tabPage;             // 自分が属しているTabPage
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MyTextBox : MFTextBox
    {
        public MyTabPage tabPage;             // 自分が属しているTabPage
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MyPictureBox : MFPictureBox
    {
        public MyTabPage tabPage;             // 自分が属しているTabPage
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MyDataGridView : MFDataGridView
    {
        public MyTabPage tabPage;
        public bool HasVerticalScrollBar()
        {
            return VerticalScrollBar.Enabled;
        }
    }
    #endregion

}
