﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using MFLib;

namespace MFLib
{
    // ■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
    // 配置が容易になるように拡張したコントロール
    //  MFLabel.mf.SetLocation
    //  MFLabel.mf.SetWidth
    //  MFLabel.mf.SetHeight
    //  MFLabel.mf.GetAroundControl
    //  この4種類の関数を使えば、たいていのレイアウトは座標計算なしで作れる

    public enum HorizontalLocation { LeftSame, RightSame, LeftOf, RightOf, CenterOf, LeftIn, RightIn };
    public enum VerticalLocation { TopSame, BottomSame, TopOf, BottomOf, CenterOf, TopIn, BottomIn };

    [System.ComponentModel.DesignerCategory("")]
    public class MFLabel : Label
    {
        public MFExtention mf;
        public MFLabel()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFButton : Button
    {
        public MFExtention mf;
        public MFButton()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFTextBox : TextBox
    {
        public MFExtention mf;
        public MFTextBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFPictureBox : PictureBox
    {
        public MFExtention mf;
        public MFPictureBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFDataGridView : DataGridView
    {
        public MFExtention mf;
        public MFDataGridView()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFTabControl : TabControl
    {
        public MFExtention mf;
        public MFTabControl()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFTabPage : TabPage
    {
        public MFExtention mf;
        public MFTabPage()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFFlowLayoutPanel : FlowLayoutPanel
    {
        public MFExtention mf;
        public MFFlowLayoutPanel()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFGroupBox : GroupBox
    {
        public MFExtention mf;
        public MFGroupBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFPanel : Panel
    {
        public MFExtention mf;
        public MFPanel()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFTableLayoutPanel : TableLayoutPanel
    {
        public MFExtention mf;
        public MFTableLayoutPanel()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFCheckBox : CheckBox
    {
        public MFExtention mf;
        public MFCheckBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFComboBox : ComboBox
    {
        public MFExtention mf;
        public MFComboBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFLinkLabel : LinkLabel
    {
        public MFExtention mf;
        public MFLinkLabel()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFListBox : ListBox
    {
        public MFExtention mf;
        public MFListBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    [System.ComponentModel.DesignerCategory("")]
    public class MFRadioButton : RadioButton
    {
        public MFExtention mf;
        public MFRadioButton()
            : base()
        {
            mf = new MFExtention(this);
        }
    }
    [System.ComponentModel.DesignerCategory("")]
    public class MFRichTextBox : RichTextBox
    {
        public MFExtention mf;
        public MFRichTextBox()
            : base()
        {
            mf = new MFExtention(this);
        }
    }

    // ■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
    // 「このボタンの右側に配置」とかいった相対的な位置・サイズ決めのための拡張
    // このクラスは直接使わないでMyButtonクラスなどを通じて使う
    // GetAroundControlだけはstatic関数なので直接使う
    public class MFExtention
    {
        Control obj;
        public MFExtention(Control obj)
        {
            this.obj = obj;
        }

        // ○○の右下、とか指定してコントロールを並べる
        // 引数
        //  obj         並べるコントロール
        //  anchor      相対位置のアンカーとなるコントロール
        //  h           水平位置の指定
        //  v           垂直位置の指定
        // 備考
        //  Anchorも、それらしいものに自動的に設定する。
        //
        // 位置指定の意味(例) 
        //  LeftSame:    左端を同じ位置に配置する。左端を揃えて、縦にボタンを並べる時とか。
        //  LeftOf:      適切なマージンをあけて、左に配置する。ボタンを左右に並べる時とか。
        //  LeftIn:      左の内側に、適切なマージンを開けて配置する。FormやPanel内に置くときとか。
        //
        // 例
        //  var b1 = new MFButton();
        //  b1.mf.SetLocation( Form1, HorizontalLocation.LeftIn, VerticalLocation.BottomIn );
        //      // これで、フォームの左下にボタンb1が配置される
        //  var b2 = new MFButton();
        //  b2.SetLocation( b1, HorizontalLocation.RightOf, VerticalLocation.TopSame );
        //      // これでb1の右横にボタンb2が配置される

        public void SetLocation(Control anchor, HorizontalLocation h, VerticalLocation v)
        {
            SetLocation(anchor, h, anchor, v);
        }

        // ○○の右、○○の下、とか2つアンカーを指定してコントロールを並べる
        // 引数
        //  obj         並べるコントロール
        //  hAnchor     水平相対位置のアンカーとなるコントロール
        //  h           水平位置の指定
        //  vAnchor     垂直相対位置のアンカーとなるコントロール
        //  v           垂直位置の指定
        // 備考
        //  Anchorも、それらしいものに自動的に設定する。
        //  hかvか、どちらかのアンカーが不要な時はnullを指定すれば位置が0に設定される
        //
        // 例
        //  上記の例に続けて
        //  var b2 = new MFButton();
        //  b3.SetLocation( b2, HorizontalLocation.LeftSame, Form1, VerticalLocation.TopIn );
        //      // これで、フォームの上端に、ボタンb2と同じ水平位置にボタンb3が表示される

        public void SetLocation(Control horizontalAnchor, HorizontalLocation h, Control verticalAnchor, VerticalLocation v)
        {
            int x = 0;
            int y = 0;
            AnchorStyles anchorStyles = AnchorStyles.None;

            if (horizontalAnchor != null)
            {
                switch (h)
                {
                    case HorizontalLocation.LeftSame:   // 同じ位置
                        x = horizontalAnchor.Location.X;
                        anchorStyles |= AnchorStyles.Left;
                        break;
                    case HorizontalLocation.LeftOf:     // 左側
                        x = horizontalAnchor.Location.X - (obj.Size.Width + obj.Margin.Right);
                        anchorStyles |= AnchorStyles.Right;
                        break;
                    case HorizontalLocation.RightOf:    // 右側
                        x = horizontalAnchor.Location.X + horizontalAnchor.Size.Width + horizontalAnchor.Margin.Right;
                        anchorStyles |= AnchorStyles.Left;
                        break;
                    case HorizontalLocation.CenterOf:   // 中央
                        x = horizontalAnchor.Location.X + (horizontalAnchor.Size.Width - obj.Size.Width) / 2;
                        anchorStyles |= AnchorStyles.Left | AnchorStyles.Right;
                        break;
                    case HorizontalLocation.LeftIn:     // 左側の内側
                        x = horizontalAnchor.Location.X + obj.Margin.Left;
                        anchorStyles |= AnchorStyles.Left;
                        break;
                    case HorizontalLocation.RightIn:     // 右側の内側
                        x = horizontalAnchor.Location.X + horizontalAnchor.Size.Width - (obj.Size.Width + obj.Margin.Right);
                        anchorStyles |= AnchorStyles.Right;
                        break;
                    case HorizontalLocation.RightSame:  // 右端に接する
                        x = horizontalAnchor.Location.X + horizontalAnchor.Size.Width - obj.Size.Width;
                        anchorStyles |= AnchorStyles.Right;
                        break;
                    default:
                        x = 0;
                        break;
                }
            }

            if (verticalAnchor != null)
            {
                switch (v)
                {
                    case VerticalLocation.TopSame:
                        y = verticalAnchor.Location.Y;
                        anchorStyles |= AnchorStyles.Top;
                        break;
                    case VerticalLocation.TopOf:
                        y = verticalAnchor.Location.Y - (obj.Height + obj.Margin.Bottom);
                        anchorStyles |= AnchorStyles.Top;
                        break;
                    case VerticalLocation.BottomOf:
                        y = verticalAnchor.Location.Y + verticalAnchor.Size.Height + verticalAnchor.Margin.Bottom;
                        anchorStyles |= AnchorStyles.Top;
                        break;
                    case VerticalLocation.CenterOf:
                        y = verticalAnchor.Location.Y + (verticalAnchor.Size.Height - obj.Size.Height) / 2;
                        anchorStyles |= AnchorStyles.Top | AnchorStyles.Bottom;
                        break;
                    case VerticalLocation.TopIn:
                        y = verticalAnchor.Location.Y + obj.Margin.Top;
                        anchorStyles |= AnchorStyles.Top;
                        break;
                    case VerticalLocation.BottomIn:
                        y = verticalAnchor.Location.Y + verticalAnchor.Size.Height - (obj.Size.Height + obj.Margin.Bottom);
                        anchorStyles |= AnchorStyles.Bottom;
                        break;
                    case VerticalLocation.BottomSame:
                        y = verticalAnchor.Location.Y + verticalAnchor.Size.Height - obj.Size.Height;
                        anchorStyles |= AnchorStyles.Bottom;
                        break;
                    default:
                        y = 0;
                        break;
                }
            }

            obj.Location = new Point(x, y);
            obj.Anchor = anchorStyles;
        }

        // 複数のアンカーの再外周コントロールを作る
        // サイズが不明のコントロールが複数個並んでいるとき、
        // その最右端に揃えて何かを置く、とかの時に使う。
        //
        // 例
        //  コントロールb1, b2, b3が存在するとき、その右端にマージンを置いてボタンを配置する
        //  Control a = MFExtention.GetAroundControl( b1, b2, b3 );
        //      // これで、b1, b2, b3にぴったり外接するコントロールaが得られる
        //  var b4 = new MFButton();
        //  b4.SetLocation( a, HorizontalLocation.RightOf, VerticalLocation.TopSame );
        //      // これで、コントロール群b1, b2, b3の右側にb4が配置される
        static public Control GetAroundControl(params Control[] anchors)
        {
            int left = int.MaxValue;
            int right = int.MinValue;
            int top = int.MaxValue;
            int bottom = int.MinValue;
            int leftMargin = int.MinValue;
            int rightMargin = int.MinValue;
            int topMargin = int.MinValue;
            int bottomMargin = int.MinValue;
            foreach (var anchor in anchors)
            {
                if (anchor.Location.X < left) left = anchor.Location.X;
                if (anchor.Location.X + anchor.Size.Width > right) right = anchor.Location.X + anchor.Size.Width;
                if (anchor.Location.Y < top) top = anchor.Location.Y;
                if (anchor.Location.Y + anchor.Size.Height > bottom) bottom = anchor.Location.Y + anchor.Size.Height;
                if (anchor.Margin.Left > leftMargin) leftMargin = anchor.Margin.Left;
                if (anchor.Margin.Right > rightMargin) rightMargin = anchor.Margin.Right;
                if (anchor.Margin.Top > topMargin) topMargin = anchor.Margin.Top;
                if (anchor.Margin.Bottom > bottomMargin) bottomMargin = anchor.Margin.Bottom;
            }

            var control = new Control();
            control.Location = new Point(left, top);
            control.Size = new Size(right - left, bottom - top);
            control.Margin = new Padding(leftMargin, topMargin, rightMargin, bottomMargin);
            return control;
        }

        // 2つのアンカーから幅を求める、Aの右側と、Bの中央、とかいう風に決める
        // 水平方向のAnchorはleft|rightに設定する
        // 
        // 位置指定の意味(例) 
        //  LeftSame:    左端
        //  LeftOf:      左端の左にマージンをとる
        //  LeftIn:      左端の右にマージンを取る
        //
        // 例
        //  コントロール x とコントロール y の間いっぱいに(マージンをとって)テキストボックスを配置する
        //  var t = new MFTextBox();
        //  t.SetLocation( x, HorizontalLocation.RightOf, VerticalLocation.TopSame );
        //  t.SetWidth( x, HorizontalLocation.RightOf, y, HorizontalLocation.LeftOf );
        public void SetWidth(Control leftAnchor, HorizontalLocation hLeft, Control rightAnchor, HorizontalLocation hRight)
        {
            int height = obj.Size.Height;
            int left = CalcX(leftAnchor, hLeft);
            int right = CalcX(rightAnchor, hRight);

            obj.Size = new Size(right - left, height);
            obj.Anchor |= AnchorStyles.Left | AnchorStyles.Right;
        }
        private int CalcX(Control anchor, HorizontalLocation h)
        {
            int x = 0;
            switch (h)
            {
                case HorizontalLocation.LeftSame:
                    x = anchor.Location.X;
                    break;
                case HorizontalLocation.LeftOf:
                    x = anchor.Location.X - anchor.Margin.Left;
                    break;
                case HorizontalLocation.LeftIn:
                    x = anchor.Location.X + obj.Margin.Left;
                    break;
                case HorizontalLocation.RightOf:
                    x = anchor.Location.X + anchor.Size.Width + anchor.Margin.Right;
                    break;
                case HorizontalLocation.RightIn:
                    x = anchor.Location.X + anchor.Size.Width - obj.Margin.Right;
                    break;
                case HorizontalLocation.RightSame:
                    x = anchor.Location.X + anchor.Size.Width;
                    break;
                case HorizontalLocation.CenterOf:
                    x = anchor.Location.X + anchor.Size.Width / 2;
                    break;
            }
            return x;
        }

        // 高さを2つのアンカーに挟まれるように自動的に決める
        // 垂直方向のAnchorはtop|bottomに設定する
        public void SetHeight(Control topAnchor, VerticalLocation vTop, Control bottomAnchor, VerticalLocation vBottom)
        {
            int width = obj.Size.Width;
            int top = CalcY(topAnchor, vTop);
            int bottom = CalcY(bottomAnchor, vBottom);

            obj.Size = new Size(width, bottom - top);
            obj.Anchor |= AnchorStyles.Top | AnchorStyles.Bottom;
        }
        private int CalcY(Control anchor, VerticalLocation v)
        {
            int y = 0;
            switch (v)
            {
                case VerticalLocation.TopSame:
                    y = anchor.Location.Y;
                    break;
                case VerticalLocation.TopOf:
                    y = anchor.Location.Y - anchor.Margin.Top;
                    break;
                case VerticalLocation.TopIn:
                    y = anchor.Location.Y + obj.Margin.Top;
                    break;
                case VerticalLocation.BottomOf:
                    y = anchor.Location.Y + anchor.Size.Height + anchor.Margin.Bottom;
                    break;
                case VerticalLocation.BottomIn:
                    y = anchor.Location.Y + anchor.Size.Height - obj.Margin.Bottom;
                    break;
                case VerticalLocation.BottomSame:
                    y = anchor.Location.Y + anchor.Size.Height;
                    break;
                case VerticalLocation.CenterOf:
                    y = anchor.Location.Y + anchor.Size.Height / 2;
                    break;
            }
            return y;
        }


    }

}
